unit Actstuff;

interface

uses
	Classes, SysUtils, Forms, WinTypes, VcbStuff, FlagDef;

Type
	MGameTable = class(TObject)
	{This is the 'list of entries' which constitutes a table.  Each item in the
	list is an entry.  It has no functions, as it is only a data structure - and
	the TList component provides us with all the required functions.}
		Table : TList;			{A Table is a TList (of MGameTableEntries)}
		TableNum : Word;
        NumEntries : Word;

		constructor Create(TableNumber : Word);
	End;	{MGameTable}

	MGameTableEntry = class(TObject)
	{This is the 'list of CondActs' which together constitute an entry in the
	Table.  Again, just a data structure, so has no functions}
		TableEntry : TList;		{A TableEntry is a TList (of CondActs)}
		Verb, Noun : Word;	{integer}
		NumCondActs : Word;

		constructor Create(Vb, Nn : Word);
	End;	{MGameTableEntry}


	MGameCondAct = class(TObject)
		CondAct : Record
			Token : Integer;
			Data1 : Word;
			Data2 : Word;
		End;	{CondAct}

		constructor Create(Token : Integer; Data1, Data2 : Word);
	End;

	Function FindTable(TableNumber : Word) : Word;
	Function GetFirstFreeTableNumber : Word;
	Function HowManyTables : Word;
	Function HowManyEntries(TableNumber : Word) : Word;
	Function GetTableNumber(ListIndex : Word) : Word;
	Function HowManyCondActs(TableNumber, EntryNumber : Word) : Word;
	Function GetVerb(TableNumber, EntryNumber : Word) : Word;
	Function GetNoun(TableNumber, EntryNumber : Word) : Word;
	Function GetCondAct(TableNumber, EntryNumber, CondActNumber : Word) :
				MGameCondAct;
	Function AddCondActToEntry(ThisEntry : MGameTableEntry;
						ThisCondAct : MGameCondAct) : Boolean;
	Function AddEntryToTable(ThisTable : MGameTable;
						ThisEntry : MGameTableEntry) : Boolean;
	Function AddTableToList(ThisList : TList;
						ThisTable : MGameTable) : Boolean;
	Function GetTableFromList(Index : Word) : MGameTable;
	Function DeleteEntryFromList(TableNo, EntryNo : Word) : Integer;
	Procedure SortActionTableList;
	Function CountMatchingEntries(TVerb, TNoun : Word) : Word;
	Function GetEntryFromVerbAndNoun(Tverb, TNoun : Word;
						EntryNumber : Word) : MGameTableEntry;
	Procedure DeleteEntryFromVerbAndNoun(Tverb, TNoun : Word; EntryNumber : Word);


implementation

Uses
	Actions;


{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * *                                                                   * * * }
{* * *                   Functions start here!                           * * * }
{* * *                                                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }




{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * *                                                                   * * * }
{* * *                   MGameCondAct functions                          * * * }
{* * *                                                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

constructor MGameCondAct.Create(Token:Integer; Data1, Data2 : Word);
{
The constructor for the class.  Just assign the input variables to the
statements in the record.

Returns:	NOTHING

Notes:		No checks are made here to see if the data is valid.  If required,
			should be made elsewhere.
}
Begin
	CondAct.Token := Token;
	CondAct.Data1 := Data1;
	CondAct.Data2 := Data2;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Constructor MGameTableEntry.Create(Vb, Nn : Word);
Begin
	TableEntry:=TList.Create;
	Verb := Vb;
	Noun := Nn;
	NumCondActs := 0;
End;	{MGameTableEntry.Create}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Constructor MGameTable.Create(TableNumber : Word);
Begin
	Table := TList.Create;
	TableNum := TableNumber;
End;	{MGameTable.Create}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

function GetFirstFreeTableNumber : Word;
{find the first free number which can be allocated to a table.
Returns:	The first number, in the range 1-65500, which is not used by a table
			ObjectNotFound - if there are no free numbers}
Var
	TableNumberArray : PChar;
	Index : Word;
	TempTable : MGameTable;
Begin
	If TableList.Count = 0 Then			{If there are no tables in the list}
	Begin								{then the first free one is always 1!}
		GetFirstFreeTableNumber := 1;
		Exit;
	End;

	GetMem(TableNumberArray,65502);

	For Index:=0 to 65500 Do
		TableNumberArray[Index] := '0';

	For Index:=0 to TableList.Count-1 Do
	Begin
		TempTable:=TableList.Items[Index];
		TableNumberArray[TempTable.TableNum]:='1';
	End;
	For Index:=1 To 65500 Do
	Begin
		If TableNumberArray[Index]='0' Then		{Found the first free}
		Begin									{number, so return it}
			GetFirstFreeTableNumber:=Index;		{and free the memory}
			FreeMem(TableNumberArray,65502);
			Exit;
		End;
	End;

	{If we get to here, then there were no spare numbers in the array}
	{So, return ObjectNotFound, to tell the user this, and free the memory}

	GetFirstFreeTableNumber := _ObjectNotFound;
	FreeMem(TableNumberArray,65502);
End;	{MGameActionTable.GetFirstFreeTableNumber}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

function FindTable(TableNumber : Word) : Word;
{Routine to check if a table with the number specified exists in the game.
Returns:	The list index of the table if a table with the number specified
				exists.
			ObjectNotFound - if no table exists with the number specified.}
Var
	Index : Word;	{local counter}
	ThisTable : MGameTable;	{Temporary variable}
Begin
	FindTable := _ObjectNotFound;		{assume failure}

	If TableList.Count = 0 Then
	Begin
		Exit;
	End;

	For Index := 0 To TableList.Count -1 Do
	Begin
		ThisTable := TableList.Items[Index];
		If ThisTable.TableNum = TableNumber Then	{Found it}
		Begin
			FindTable := Index;
			Break;	{Index := TableList.Count -1;}		{exit from the loop}
		End;	{If}
	End;	{For}
End;	{FindTable}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * * Private functions.  These are designed to provide a standard      * * * }
{* * *   interface to the tables, by returning specific information      * * * }
{* * *   about the tables, and the entries therein.  See individual      * * * }
{* * *   routines for more information                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

function HowManyTables : Word;
{Routine to see how many tables there are in the game.
Returns:	The number of tables in the game - NumTables}
Begin
	HowManyTables := TableList.Count;
End;	{HowManyTables}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

function GetTableNumber(ListIndex : Word) : Word;
{function the return the table number of the table at the specified list index.
	E.g. if the tables in the list were the following:
			4, 8, 13, 67, 345
	Then GetTableNumber(3) would return 13
Returns:	The table number of the table at the index specified.
			ObjectNotFound - If the index is out of range}
Var
	ThisTable : MGameTable;
Begin
	If ListIndex > HowManyTables Then
	Begin							{if the list index is greater than the}
		GetTableNumber := _ObjectNotFound;	{number of tables, then there is no table}
		Exit;						{at that index...return error}
	End;	{If}

	ThisTable := TableList.Items[ListIndex];

	GetTableNumber := ThisTable.TableNum;
End;	{GetTableNumber}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

function HowManyEntries(TableNumber : Word) : Word;
{Function to determine how many entries there are in a particular table
Returns:	The number of entries in the specified table.
			ObjectNotFound - if the table does not exist.
NOTE:	The first entry in the list is used to hold the verb and noun for the
			entry, so should be ignored.}
Var
	ListIndex : Word;	{used to hold the index of the requested table in the
							table list}
	ThisTable : MGameTable;	{used to store a temporary copy of the head node of
								the requested table}
Begin
	ListIndex := FindTable(TableNumber);	{get the position of the required
											table in the table list}
	If (ListIndex = _ObjectNotFound) Then
	Begin							{if the table does not exist, return error}
		HowManyEntries := _ObjectNotFound;
		Exit;
	End;

	{first, 'extract' the table required from the list}

	ThisTable := TableList.Items[ListIndex];

	{next, count the number of items in the 'table' list - that is the number of
		entries in the table}

	HowManyEntries := ThisTable.NumEntries;
End;	{HowManyEntries}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function HowManyCondActs(TableNumber, EntryNumber : Word) : Word;
{Function to return how many CondActs a specific entry in a specific table has.
Returns:	The number of condacts in the specified entry of the specified
				table.
			ObjectNotFound - if the table number specified is invalid.
			65534 - if the entry number is invalid.}
Var
	ListIndex : Word;
	ThisTable : MGameTable;
	ThisEntry : MGameTableEntry;
Begin
	ListIndex := FindTable(TableNumber);

	If (ListIndex = _ObjectNotFound) Then		{if the table does not exist, return error}
	Begin
		HowManyCondActs := _ObjectNotFound;
		Exit;
	End;

	If EntryNumber > HowManyEntries(TableNumber) Then
	Begin							{if there are not enough entries, return}
		HowManyCondActs:= 65534;		{error}
		Exit;
	End;

	ThisTable := TableList.Items[ListIndex];			{get table}

	ThisEntry := ThisTable.Table.Items[EntryNumber];	{get entry}

	HowManyCondActs := ThisEntry.NumCondActs;			{count condacts}
End;	{HowManyCondActs}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetVerb(TableNumber, EntryNumber : Word) : Word;	{returns integer!}
{Function to return the verb of the specified entry in the specified table
Returns:	The verb (value) of the specified entry in the specified table.
			ObjectNotFound - if the table does not exist.
			65534 - if the entry does not exist.}
Var
	ListIndex : Word;
	ThisTable : MGameTable;
	ThisEntry : MGameTableEntry;
Begin
	ListIndex := FindTable(TableNumber);

	If (ListIndex = _ObjectNotFound) Then		{if the table does not exist, return error}
	Begin
		GetVerb := _ObjectNotFound;
		Exit;
	End;

	If (EntryNumber > HowManyEntries(TableNumber)) Then
	Begin							{if there are not enough entries, return}
		GetVerb := 65534;			{error}
		Exit;
	End;

	ThisTable := TableList.Items[ListIndex];			{get table}

	ThisEntry := ThisTable.Table.Items[EntryNumber];	{get entry}

	GetVerb := ThisEntry.Verb;							{get verb}
End;	{GetVerb}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetNoun(TableNumber, EntryNumber : Word) : Word;	{returns integer!}
{Function to return the noun of the specified entry in the specified table
Returns:	The noun (value) of the specified entry in the specified table.
			ObjectNotFound - if the table does not exist.
			65534 - if the entry does not exist.}
Var
	ListIndex : Word;
	ThisTable : MGameTable;
	ThisEntry : MGameTableEntry;
Begin
	ListIndex := FindTable(TableNumber);

	If (ListIndex = _ObjectNotFound) Then		{if the table does not exist, return error}
	Begin
		GetNoun := _ObjectNotFound;
		Exit;
	End;

	If EntryNumber > HowManyEntries(TableNumber) Then
	Begin							{if there are not enough entries, return}
		GetNoun := 65534;		{error}
		Exit;
	End;

	ThisTable := TableList.Items[ListIndex];			{get table}

	ThisEntry := ThisTable.Table.Items[EntryNumber];	{get entry}

	GetNoun := ThisEntry.Noun;							{get noun}
End;	{GetNoun}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetCondAct(TableNumber, EntryNumber, CondActNumber : Word) : MGameCondAct;
{Function to return the specified condact, in the specified entry, of the
	specified table.
Returns:	The condact in the specified position in the specified entry of the
				specified table.
			Token as -1 - if the table does not exist.
			Token as -2 - if the entry does not exist.}
Var
	ListIndex : Word;
	ThisTable : MGameTable;
	ThisEntry : MGameTableEntry;
Begin
	ListIndex := FindTable(TableNumber);

	If (ListIndex = _ObjectNotFound) Then		{if the table does not exist, return error}
	Begin
		GetCondAct.CondAct.Token := -1;
		Exit;
	End;

	If EntryNumber > HowManyEntries(TableNumber) Then
	Begin									{if there are not enough entries,}
		GetCondAct.CondAct.Token := -2;		{return error}
		Exit;
	End;

	ThisTable := TableList.Items[ListIndex];					{get table}

	ThisEntry := ThisTable.Table.Items[EntryNumber];			{get entry}

	GetCondAct := ThisEntry.TableEntry.Items[CondActNumber];	{get condact}
End;	{GetCondAct}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function AddCondActToEntry(ThisEntry : MGameTableEntry;
					ThisCondAct : MGameCondAct) : Boolean;
{Function to add a condact to an entry.
Returns:	True if the condact was added successfully.
			False if the entry already has 65500 condacts (unlikely).}
Begin
	If ThisEntry.TableEntry.Count <> 65500 Then
	Begin
		ThisEntry.TableEntry.Add(ThisCondAct);
		Inc(ThisEntry.NumCondActs);
		AddCondActToEntry := True;
	End Else Begin
		AddCondActToEntry := False;
	End;	{If}
End;	{AddCondActToEntry}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function AddEntryToTable(ThisTable : MGameTable;
					ThisEntry : MGameTableEntry) : Boolean;
{Function to add an entry to a table.  Updates the 'numentries' field.  For
	compatibility and futureproofing, all additions should be done using this
	function.
Returns:	True if the entry was added successfully.
			False if the table already has 65500 entries (unlikely, but
				possible).}
Begin
	If ThisTable.NumEntries <> 65500 Then
	Begin
		ThisTable.Table.Add(ThisEntry);
		ThisTable.NumEntries := ThisTable.NumEntries + 1;
		AddEntryToTable := True
	End Else Begin
		AddEntryToTable := False;
	End;	{If}
End;	{AddEntryToTable}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function AddTableToList(ThisList : TList; ThisTable : MGameTable) : Boolean;
{Function to add a table to a list.  Updates the 'tablecount' field.
Returns:	True if the table added successfully.
			False if the table list already has 65500 tables.}
Begin
	ThisList.Add(ThisTable);
	Inc(TableCount);
End;	{AddTableToList}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetTableFromList(Index : Word) : MGameTable;
Begin
	GetTableFromList := TableList.Items[Index];
End;	{GetTableFromList}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function DeleteEntryFromList(TableNo, EntryNo : Word) : Integer;
{delete the specified entry from the specified table.
Returns:	-1 if the table list index is invalid
			-2 if the entry index is invalid
			0 if no error}
Var
	ThisTable : MGameTable;
Begin
	If TableNo > TableList.Count -1 Then
	Begin
		DeleteEntryFromList := -1;	{table out of range}
		Exit;
	End;

	ThisTable := TableList.Items[TableNo];

	If EntryNo > ThisTable.NumEntries Then
	Begin
		DeleteEntryFromList := -2;	{entry out of range}
		Exit;
	End;

	ThisTable.Table.Delete(EntryNo);
	Dec(ThisTable.NumEntries);

	DeleteEntryFromList := 0;	{no error}
End;	{DeleteEntryFromList}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Procedure SortActionTableList;
{firstly, sort the tables into ascending numerical order, and then sort the
entries in the table into ascending numerical order on verb number, and then on
noun number.  If the verb and noun are the same for two or more entries, then
their order in the table will be the same order they were entered into the
table.}
Var
	ThisTable : MGameTable;
	TableCounter : Word;	{Integer!}

	Function SortTables : Boolean;
	Var
		DoneASort : Boolean;
		Index : Word;	{Integer!}
		ThisTable : MGameTable;
		NextTable : MGameTable;
	Begin
		If ((TableList.Count = 0) Or (TableList.Count = 1)) Then
		{if the table list has 0 or 1 tables, then it is always sorted}
		Begin
			SortTables := False;
			Exit;
		End;	{If}

		DoneASort := False;

		For Index := 0 to TableList.Count -2 Do
		Begin
			ThisTable := TableList.Items[Index];
			NextTable := TableList.Items[Index+1];

			If ThisTable.TableNum > NextTable.TableNum Then
			Begin
				TableList.Exchange(Index, Index+1);
				DoneASort := True;
			End;	{If}
		End;	{For}

		If DoneASort = True Then SortTables := True
		Else SortTables := False;
	End;	{sorttables}

	Function SortEntries(ThisTable : MGameTable) : Boolean;
	Var
		DoneASort : Boolean;
		Index : Word;	{Integer!}
		ThisEntry : MGameTableEntry;
		NextEntry : MGameTableEntry;
	Begin
		If ((ThisTable.NumEntries = 0) Or (ThisTable.NumEntries = 1)) Then
		Begin
			SortEntries := False;
			Exit;
		End;

		DoneASort := False;

		For Index := 0 to ThisTable.NumEntries -2 Do
		Begin
			ThisEntry := ThisTable.Table.Items[Index];
			NextEntry := ThisTable.Table.Items[Index+1];

			If (ThisEntry.Verb > NextEntry.Verb) Then
			Begin
				ThisTable.Table.Exchange(Index, Index+1);
				DoneASort := True
			End Else Begin
				If ((ThisEntry.Verb = NextEntry.Verb) And
					(ThisEntry.Noun > NextEntry.Noun)) Then
				Begin
					ThisTable.Table.Exchange(Index, Index+1);
					DoneASort := True;
				End;	{if verbnouns}
			End;	{if verbs}
		End;	{For}

		If DoneASort = True Then SortEntries := True
		Else SortEntries := False;
	End;	{SortEntries}

Begin
	Repeat
	Until SortTables = False;
	If TableList.Count <> 0 Then
	Begin
		For TableCounter := 0 To TableList.Count -1 Do
		Begin
			ThisTable := TableList.Items[TableCounter];
			Repeat
			Until SortEntries(ThisTable) = False;
		End;	{For}
	End;	{If}
End;	{SortActionTableList}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function CountMatchingEntries(TVerb, TNoun : Word) : Word;
{count the number of entries in the CURRENT table whose verb is TVerb and whose
noun is TNoun, return the count}
Var
	ListIndex : Word;	{position in table list of current table}
	ThisTable : MGameTable;	{the current table we are working with}
	Index : Word;	{counter}
	EntryCounter : Word;	{count of matches}
	ThisEntry : MGameTableEntry;	{temporary variable}
Begin
	EntryCounter := 0;
	ListIndex := FindTable(CurrentTable);
	If ListIndex = _ObjectNotFound Then
	Begin
		CountMatchingEntries := 0;	{no matches in non-existent table}
		Exit;
	End;

	ThisTable := TableList.Items[ListIndex];

	If ThisTable.NumEntries = 0 Then
	Begin
		CountMatchingEntries := 0;	{no matches in an empty table}
		Exit;
	End;

	For Index := 0 To ThisTable.NumEntries -1 Do
	Begin
		ThisEntry := ThisTable.Table.Items[Index];
		If ((ThisEntry.Verb = TVerb) And (ThisEntry.Noun = TNoun)) Then
		Begin
			EntryCounter := EntryCounter + 1;
		End;	{if}
	End;	{For}

	CountMatchingEntries := EntryCounter;
End;	{CountMatchingEntries}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetEntryFromVerbAndNoun(Tverb, TNoun : Word; EntryNumber : Word) :
									MGameTableEntry;
{given the verb and noun for an entry, return the specified occurrence of that
entry.  Note: check to make sure that there are enough entries before calling
this routine.  I.e. if you ask for entry number 5, and there are only 4, result
will be a table entry with verb, noun and numentries set to 0.}
Var
	ListIndex : Word;	{position in table list of current table}
	ThisTable : MGameTable;	{the current table we are working with}
	Index : Word;	{counter}
	ThisTableEntry : MGameTableEntry;	{the entry we are going to return}
Begin
	ListIndex := FindTable(CurrentTable);
	If ListIndex = _ObjectNotFound Then
	Begin
		ThisTableEntry := MGameTableEntry.Create(0,0);
		GetEntryFromVerbAndNoun := ThisTableEntry;
		Exit;
	End;

	ThisTable := TableList.Items[ListIndex];

	If ThisTable.NumEntries = 0 Then
	Begin
		ThisTableEntry := MGameTableEntry.Create(0,0);
		GetEntryFromVerbAndNoun := ThisTableEntry;
		Exit;
	End;

	For Index := 0 To ThisTable.NumEntries -1 Do
	Begin
		ThisTableEntry := ThisTable.Table.Items[Index];
		If ((ThisTableEntry.Verb = TVerb) And (ThisTableEntry.Noun = TNoun)) Then
		Begin
			EntryNumber := EntryNumber -1;
			If EntryNumber = 0 Then
			Begin
				GetEntryFromVerbAndNoun := ThisTableEntry;
				Exit;
			End;	{nested if}
		End;	{if}
	End;	{For}
End;	{GetEntryFromVerbAndNoun}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Procedure DeleteEntryFromVerbAndNoun(Tverb, TNoun : Word; EntryNumber : Word);
{given the verb and noun for an entry and the specified occurrence required,
delete that occurrence}
Var
	ListIndex : Word;	{position in table list of current table}
	ThisTable : MGameTable;	{the current table we are working with}
	Index : Word;	{counter}
	ThisTableEntry : MGameTableEntry;	{the entry we are going to return}
Begin
	ListIndex := FindTable(CurrentTable);
	If ListIndex = _ObjectNotFound Then
		Exit;

	ThisTable := TableList.Items[ListIndex];

	If ThisTable.NumEntries = 0 Then
		Exit;

	For Index := 0 To ThisTable.NumEntries -1 Do
	Begin
		ThisTableEntry := ThisTable.Table.Items[Index];
		If ((ThisTableEntry.Verb = TVerb) And (ThisTableEntry.Noun = TNoun)) Then
		Begin
			EntryNumber := EntryNumber -1;
			If EntryNumber = 0 Then
			Begin
				ThisTable.Table.Delete(Index);
				TableList.Delete(ListIndex);
				ThisTable.NumEntries := ThisTable.NumEntries -1;
				AddTableToList(TableList, ThisTable);
				Exit;
			End;	{nested if}
		End;	{if}
	End;	{For}
End;	{DeleteEntryFromVerbAndNoun}

End.	{ActStuff}
