Unit Objstuff;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
{* * *                                                                   * * * }
{ * * *  The purpose of this unit is to provide the routines needed for   * * *}
{* * *    the handling of objects in the game - creating, deleting etc.  * * * }
{ * * *                                                                   * * *}
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Interface

Uses
	Classes, SysUtils, Forms, WinTypes, NewObj;

Type
	GameObject = Class(TObject)
		ObjNumber : Word;				{The number of the object}
		ObjShortName : String[50];		{The object's short name}
		ObjStartingLoc : Word;			{The object's starting location}
		ObjPresentString : String[100];	{The string shown when the object is present}
		ObjNoun : Word;					{The number of the corresponding noun}
		ObjAdjective : Word;			{The number of the corresponding adjective}
		ObjWeight : Word;				{The weight of the object}
		ObjValue : Word;				{the value of the object}
		ObjContainer : Boolean;			{Is it a container?}
		ObjWearable : Boolean;			{Is it wearable?}
		ObjInvisible : Boolean;			{Does it show up on location desribe?}
		ObjDescLength : Word;			{The length of the description}
		ObjDescription : Array[0..32001] of Char;	{The object's description}

		ObjCurrentLocation : Word;			{used only in the game - not by the
										designer.  holds the current location of
										the object.  It is set to be the same as
										the object's starting location when the
										game starts}

		Constructor Create(Number, StartingLoc, Noun, Adjective, Weight, Value : Word;
					ShortName, PresentString : String;
					Container, Wearable, Invisible : Boolean;
					DescLength : Word;
					Description : Array of Char);	{create our constructor}
	End;

	MGameObject = Class(TObject)
		Private
			ObjectList : TList;	{have to overload the class definition}
								{for TObject, to that a TList, i.e. our}
								{list of Objects, will hold an Object}
								{as each node.  NOTE - have to call it Obj,}
								{because 'object' is a eserved word}
		Public
			NumberOfObjects : Word;

		Constructor Create;	{create our constructor}
		Function FindObject(Number : Word) : Boolean;
		Function GetFirstFreeNumber : Word;
		Function NumObjects : Word;
		Procedure SortList;
		Function AddObject(
					Number, StartingLoc, Noun, Adjective, Weight, Value : Word;
					ShortName, PresentString : String;
					Container, Wearable, Visible : Boolean;
					DescLength : Word;
					Description : Array of Char) : Integer;
		Function DeleteObject(ObjectNumber : Word) : Boolean;
		Function GetObject(ObjectNumber : Word) : GameObject;
		Function GetObjectAtIndex(ListIndex : Word) : GameObject;
		Function ResetNounAtIndex(ListIndex : Word) : Boolean;
		Function ResetAdjectiveAtIndex(ListIndex : Word) : Boolean;
		Function ResetStartingLocAtIndex(ListIndex : Word) : Boolean;

		Function GetObjectLocation(ObjNum : Word) : Word;
		Procedure SetObjectLocation(ObjNum, LocNum : Word);
		Function CheckNoun(NounVal : Word) : Boolean;
		Function GetObjectFromNnAndAdj(NounVal, AdjVal : Word) : Word;

	End;	{MGameObject}

		{*********************************************}

		{*********************************************}
		{**           Exported Procedures           **}
		{*********************************************}

		{*********************************************}

		procedure ResetDialogBox;	{reset the newObjDlg to defaults}

implementation

uses
	ObjMenu, VocabMnu, VcbStuff;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * *                                                                   * * * }
{* * *                   Functions start here!                           * * * }
{* * *                                                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }




{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * *                                                                   * * * }
{* * *                  Class GameObject functions                       * * * }
{* * *                                                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }


{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Constructor GameObject.Create(
					Number, StartingLoc, Noun, Adjective, Weight, Value : Word;
					ShortName, PresentString : String;
					Container, Wearable, Invisible : Boolean;
					DescLength : Word;
					Description : Array of Char);
Var
	Index : Word;
Begin
	ObjNumber := Number;
	ObjShortName := ShortName;
	ObjStartingLoc := StartingLoc;
	ObjPresentString := PresentString;
	ObjDescLength := DescLength;
	For Index := 0 to ObjDescLength Do
		ObjDescription[Index] := Description[Index];
	ObjNoun := Noun;
	ObjAdjective := Adjective;
	ObjWeight := Weight;
	ObjValue := Value;
	ObjContainer := Container;
	ObjWearable := Wearable;
	ObjInvisible := Invisible;

	ObjCurrentLocation := StartingLoc;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * *                                                                   * * * }
{* * *                   Class MGameObject functions                     * * * }
{* * *                                                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }


{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Constructor MGameObject.Create;
{create the list, and initialise the number of objects to zero}
Begin
	ObjectList := TList.Create;
	NumberOfObjects := 0;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.FindObject(Number : Word) : Boolean;
{Checks if the object number specified is used already.
Returns:	TRUE if so
			FALSE if not}
Var
	ThisObject : GameObject;	{temp var}
	Index : Word;		{counter}
	Found : Boolean;		{local copy of true/false}
	ObjNumber : Word;
Begin
	Found := False;		{assume failure}
	If NumObjects <> 0 Then
	Begin
		For Index:=0 to ObjectList.Count -1 Do	{loop through list, check each obj}
		Begin
			ThisObject := ObjectList.Items[Index];
			ObjNumber := ThisObject.ObjNumber;
			If ObjNumber = Number Then
			Begin
				Found := True;	{found it}
				Break;	{Exit the loop}
			End;	{if found}
		End;	{for}
	End;	{it at least one object}

	FindObject := Found;	{return our true/false status}
End;	{FindObject}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.GetFirstFreeNumber : Word;
{Using the List of objects, return the first available object number, in the
	range 0-65500
Returns:	The first unused number or
			65535 if there are no free numbers}
Var
	ObjNumberArray : PChar;
	Index : Word;
	TempObject : GameObject;
Begin
	If NumObjects = 0 Then
	Begin
		GetFirstFreeNumber := 1;
		Exit;
	End;

	GetMem(ObjNumberArray,65502);

	For Index:=0 to 65500 Do
		ObjNumberArray[Index] := '0';

	For Index:=0 to ObjectList.Count-1 Do
	Begin
		TempObject:=ObjectList.Items[Index];
		ObjNumberArray[TempObject.ObjNumber]:='1';
	End;
	For Index:=1 To 65500 Do
	Begin
		If ObjNumberArray[Index]='0' Then
		Begin
			GetFirstFreeNumber:=Index;
			FreeMem(ObjNumberArray,65502);
			Exit;
		End;
	End;

	FreeMem(ObjNumberArray,65502);

	{If we get to here then there are no free numbers}
	GetFirstFreeNumber := 65535;
End;	{GetFirstFreeNumber}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.NumObjects : Word;
{get the number of objects in the list}
Begin
	NumObjects := NumberOfObjects;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Procedure MGameObject.SortList;

	function Sort : Boolean;
	var
		Done : Boolean;
		Index : Word;
		ThisObject, NextObject : GameObject;
	Begin
		Done := False;
		If (NumObjects = 0) or (NumObjects = 1) Then
			Sort := False	{A list with 0 or 1 items is always sorted}
		Else
		Begin
			For Index:=0 to ObjectList.Count-2 Do
			Begin
				ThisObject := ObjectList.Items[Index];
				NextObject := ObjectList.Items[Index+1];
				If ThisObject.ObjNumber > NextObject.ObjNumber Then
				Begin
					ObjectList.Exchange(Index, Index+1);
					Done := True;	{To say that we exchanged at least one item}
				End;
			End;
		End;
		Sort := Done;
	End;

Begin
	repeat
	Until Sort = False;
End;	{SortList}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.AddObject(Number, StartingLoc, Noun, Adjective, Weight, Value : Word;
	ShortName, PresentString : String; Container, Wearable, Visible : Boolean;
	DescLength : Word; Description : Array of Char) : Integer;
{add a new object to the list
Returns:	0 if the add operation went ok
			-1 if there are already the maximum number of objects in the game
			-2 If there was already an object in the game with the number
				specified}
Var
	ThisObject : GameObject;
Begin
	If NumObjects = 65500 Then
	Begin
		AddObject := -1;
		Exit;
	End;
	If (FindObject(Number) = True) Then
	Begin
		AddObject := -2;
		Exit;
	End;
	ThisObject := GameObject.Create(Number, StartingLoc, Noun, Adjective,
		Weight, Value, ShortName, PresentString, Container, Wearable, Visible,
		DescLength, Description);
	ObjectList.Add(ThisObject);
	SortList;
	AddObject := 0;
	NumberOfObjects := NumberOfObjects + 1;
End;	{AddObject}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.DeleteObject(ObjectNumber : Word) : Boolean;
Var
	Done : Boolean;
	Index : Word;
	ThisObject : GameObject;
Begin
	Done := False;
	For Index := 0 to NumObjects -1 Do
	Begin
		ThisObject := ObjectList.Items[Index];
		If ThisObject.ObjNumber = ObjectNumber Then
		Begin
			Done := True;
			ObjectList.Delete(Index);
			NumberOfObjects := NumberOfObjects -1;
			Break;
		End;
	End;
	DeleteObject := Done;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.GetObject(ObjectNumber : Word) : GameObject;
Var
	Index : Word;
	Done : Boolean;
	ThisObject : GameObject;
Begin
	Done := False;
	For Index := 0 To NumObjects -1 Do
	Begin
		ThisObject := ObjectList.Items[Index];
		If ThisObject.ObjNumber = ObjectNumber Then
		Begin
			Done := True;
			GetObject := ThisObject;
			Break;
		End;
	End;
	If Done = False Then
		GetObject := NIL;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.GetObjectAtIndex(ListIndex : Word) : GameObject;
{return the object at the specified list index
Returns:	the object at the specified list index
			NIL if the list index is invalid}
Var
	ThisObject : GameObject;
Begin
	If ListIndex >= ObjectList.Count Then
	Begin
		GetObjectAtIndex := NIL;
		Exit;
	End;

	ThisObject := ObjectList.Items[listIndex];
	GetObjectAtIndex := ThisObject;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.ResetNounAtIndex(ListIndex : Word) : Boolean;
Var
	ThisObject : GameObject;
Begin
	ThisObject := ObjectList.Items[ListIndex];
	ThisObject.ObjNoun := 0;
	ObjectList.Delete(ListIndex);
	ObjectList.Insert(ListIndex, ThisObject);
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.ResetAdjectiveAtIndex(ListIndex : Word) : Boolean;
Var
	ThisObject : GameObject;
Begin
	ThisObject := ObjectList.Items[ListIndex];
	ThisObject.ObjAdjective := 0;
	ObjectList.Delete(ListIndex);
	ObjectList.Insert(ListIndex, ThisObject);
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.ResetStartingLocAtIndex(ListIndex : Word) : Boolean;
Var
	ThisObject : GameObject;
Begin
	ThisObject := ObjectList.Items[ListIndex];
	ThisObject.ObjStartingLoc := 0;
	ObjectList.Delete(ListIndex);
	ObjectList.Insert(ListIndex, ThisObject);
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.GetObjectLocation(ObjNum : Word) : Word;
{returns the location of the specified object
Returns:	the object's location
			65535 if the object does not exist}
Var
	ThisObject : GameObject;
	Index : Integer;
Begin
	For Index := 0 To NumObjects -1 Do
	Begin
		ThisObject := ObjectList.Items[Index];
		If ThisObject.ObjNumber = ObjNum Then
		Begin
			GetObjectLocation := ThisObject.ObjCurrentLocation;
			Exit;
		End;
	End;
	{If we get to here, then the object is not found}
	GetObjectLocation := 65535;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Procedure MGameObject.SetObjectLocation(ObjNum, LocNum : Word);
{set the current location of object Objnum to be location LocNum}
Var
	Index : Word;
	ThisObject : GameObject;
Begin
	For Index := 0 to NumObjects -1 Do
	Begin
		ThisObject := ObjectList.Items[Index];
		If ThisObject.ObjNumber = ObjNum Then
		Begin
			ObjectList.Delete(Index);	{delete the object}
			ThisObject.ObjCurrentLocation := LocNum;	{set the current location}
			ObjectList.Insert(Index, ThisObject);	{put it back}
			SortList;	{just to be sure}
			Break;	{exit the loop}
		End;
	End;
End;	{SetObjectLocation}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.CheckNoun(NounVal : Word) : Boolean;
Var
	Index : Word;
	ThisObject : GameObject;
	Found : Boolean;
Begin
	Found := False;	{assume failure}
	For Index := 0 To NumObjects -1 Do
	Begin
		ThisObject := ObjectList.Items[Index];
		If ThisObject.ObjNoun = NounVal Then
		Begin
			Found := True;
			Break;
		End;
	End;
	CheckNoun := Found;
End;	{CheckNoun}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameObject.GetObjectFromNnAndAdj(NounVal, AdjVal : Word) : Word;
{figure out the number of the object which has the noun NounVal and the
adjective AdjVal associated with it.
Returns:	the object's number
			65535 if no object found with that noun and adjective}
Var
	Index : Word;
	Found : Word;
	ThisObject : GameObject;
Begin
	Found := 65535;	{Assume failure}
	For Index := 0 To NumObjects -1 Do
	Begin
		ThisObject := ObjectList.Items[Index];
		If (((ThisObject.ObjNoun = NounVal)) And
			((ThisObject.ObjAdjective = AdjVal) Or (AdjVal = 0))) Then
		Begin
			Found := ThisObject.ObjNumber;
			Break;
		End;
	End;
	GetObjectFromNnAndAdj := Found;
End;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * *                                                                   * * * }
{* * *                         General functions                         * * * }
{* * *                                                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }


{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Procedure ResetDialogBox;
Begin
	NewObjDlg.Caption := 'New Object';
	NewObjDlg.ShortName.Text := '';
	NewObjDlg.StartLoc.Value := 0;
	NewObjDlg.ObjPresent.Text := '';
	NewObjDlg.Memo1.Clear;
	NewObjDlg.NounBox.Text := '';
	NewObjDlg.AdjectiveBox.Text := '';
	NewObjDlg.Container.Checked := False;
	NewObjDlg.Invisible.Checked := False;
	NewObjDlg.Wearable.Checked := False;
	NewObjDlg.WeightBox.Value := 0;
	NewObjDlg.ValueBox.Value := 0;

	NewObjDlg.ShortName.ReadOnly := False;
	NewObjDlg.ObjPresent.ReadOnly := False;
	NewObjDlg.Memo1.ReadOnly := False;
	NewObjDlg.StartLoc.ReadOnly := False;
	NewObjDlg.WeightBox.ReadOnly := False;
	NewObjDlg.ValueBox.ReadOnly := False;
	NewObjDlg.NounBox.Enabled := True;
	NewObjDlg.AdjectiveBox.Enabled := True;
	NewObjDlg.Container.Enabled := True;
	NewObjDlg.Invisible.Enabled := True;
 	NewObjDlg.Wearable.Enabled := True;
End;

End.	{ObjStuff}
