unit Text2Num;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
{* * *                                                                   * * * }
{ * * *  The purpose of this unit is to provide function which will       * * *}
{* * *    the input of the user - it takes a table entry, no matter      * * * }
{ * * *   how long, and will convert it into the integer form required.   * * *}
{* * *   The routines here perform checking to ensure that, e.g. a token * * * }
{ * * *   has the correct amount of data following it, and that the data  * * *}
{* * *    are valid - so if there is no location 50, 'AT 50' will return * * * }
{ * * *   an error.                                                       * * *}
{* * *                                                                   * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }

interface

Uses
	Classes, SysUtils, LocStuff, ObjStuff, MesStuff, IntStuff, Controls,
	ActStuff, Globals, VcbStuff, FlagDef;

Const
	NumFlags = 256;			{The number of 'flags' (variables) in the game}
	NumBulbs = 256;			{The number of 'bulbs' (boolean vars) in the game}
	NumConditions = 32;     {There are 32 possible conditions in the game.
							These are enumerated in the 'Conditions' array}
	Conditions : Array[1..NumConditions] Of String[11] = (
				'QUIT',

				'AT', 'NOTAT', 'ATGREATER', 'ATLESS', 'PRESENT',
				'ABSENT', 'WEARING', 'NOTWEARING', 'CARRYING', 'NOTCARRYING',
				'ZERO', 'NOTZERO', 'ON', 'OFF', 'ADJECT1',
				'ADVERB', 'PREPOSITION', 'NOUN2', 'ADJECT2', 'CHANCE',

				'ATBETWEEN', 'ISAT', 'ISNOTAT', 'EQ', 'NOTEQ',
				'GREATER', 'LESSTHAN', 'SAME', 'NOTSAME', 'LESS',
				'MORE');

	NumActions = 64;        {There are 64 actions in the game.  These are
							enumerated in the 'Actions' array}
	Actions : Array[1..NumActions] Of String[11] = (
				'DROPALL', 'AUTOG', 'AUTOD', 'AUTOW', 'AUTOR',
				'WHATO', 'TURNS', 'SCORE', 'CLS', 'NEWLINE',
				'LISTOBJ', 'INVEN', 'DESC', 'DONE', 'NOTDONE',
				'END', 'SAVE', 'LOAD', 'ANYKEY', 'NEWTEXT',

				'GET', 'DROP', 'WEAR', 'REMOVE', 'CREATE',
				'DESTROY', 'PUTO', 'AUTOP', 'AUTOT', 'SET',
				'CLEAR', 'TURNON', 'TURNOFF', 'RANDOM', 'MOVE',
				'GOTO', 'WEIGHT', 'PRINT', 'MES', 'MESSAGE',
				'INTMESS', 'BG', 'FG', 'LISTAT', 'PAUSE',
				'TABLE', 'DOALL',

				'SWAP', 'PLACE', 'PUTIN', 'TAKEOUT', 'COPYOO',
				'COPYOF', 'COPYFO', 'PLUS', 'MINUS', 'ADD',
				'SUB', 'COPYFF', 'BEEP', 'ABILITY',

				'EXAMINE', 'AUTOE', 'LET');

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
{* * *                                                                   * * * }
{ * * *                  CondDataVal and ActDataVal                       * * *}
{* * *                                                                   * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * *                                                                   * * *}
{* * *   These two arrays are used to define wnat data each CondAct      * * * }
{ * * *   takes.  When it is found out how much data each CondAct is      * * *}
{* * *    followed by, it is necessary to determine that the CondAct is  * * * }
{ * * *   followed by valid data, i.e. that the numbers refer to actual   * * *}
{* * *    data in the game.  The different checks are expressed by a     * * * }
{ * * *   1-digit code, which refers to the type of data.  See table      * * *}
{* * *    below for explanation of codes.                                * * * }
{ * * *                                                                   * * *}
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
{* * *                                                                   * * * }
{ * * *			Locations			1									  * * *}
{* * *			Messages			2									 * * * }
{ * * *			Internal Messages	3									  * * *}
{* * *			Objects				4									 * * * }
{ * * *			Table Number		5									  * * *}
{* * * 			Number				6									 * * * }
{ * * *			Flag				7									  * * *}
{* * *			Bulb				8									 * * * }
{ * * *			Word Value			9									  * * *}
{* * *                                                                   * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }

	CondDataVal : Array[1..NumConditions] of String[2] = (
				'00',

				'01', '01', '01', '01', '04',
				'04', '04', '04', '04', '04',
				'07', '07', '08', '08', '09',
				'09', '09', '09', '09', '06',

				'11', '41', '41', '76', '76',
				'76', '76', '77', '77', '77',
				'77');

	ActDataVal : Array[1..NumActions] of String[2] = (
				'00', '00', '00', '00', '00',
				'00', '00', '00', '00', '00',
				'00', '00', '00', '00', '00',
				'00', '00', '00', '00', '00',

				'04', '04', '04', '04', '04',
				'04', '01', '01', '01', '07',
				'07', '08', '08', '07', '07',
				'01', '07', '07', '02', '02',
				'03', '06', '06', '04', '06',
				'05', '01',

				'44', '41', '41', '41', '44',
				'47', '74', '76', '76', '77',
				'77', '77', '66', '66',

				'04', '00', '76');

Type
	MGameInputWord = class(TObject)
		Token : String[11];     {Overload TObject's data, so that we can hold
							a string as the object of the node in a TList}
	constructor
		Create(Value : String);
	End;    {MGameInputWord}






Function IsAction(Token : String) : Integer;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
Function IsCondition(Token : String) : Integer;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
Function IsCondAct(Token : String) : Integer;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
Function StringToList(InputString : PChar) : TList;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
Function ProcessInput(Input : TCaption; Var ErrorPosn : Integer;
						Verb, Noun : Word) : MGameTableEntry;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
Function ValidateData(IsAction : Boolean; TokenValue : Integer;
						NumData, Data1, Data2 : Integer) : Boolean;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
Function TokensToIntegers(Input : TList; Verb, Noun : Word) : MGameTableEntry;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }


implementation

uses
	Debug, IntDlg;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * *                                                                   * * * }
{* * *                   Functions start here!                           * * * }
{* * *                                                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }




{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
Constructor MGameInputWord.Create(Value : String);
{Note - No checks are made on the validity of data here}
Begin
	Token := Value;
End;    {Create}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsCondition(Token : String) : Integer;
{Takes a string as input and returns TRUE if it is a valid condition token,
or false otherwise.

NOTE - input string must be in upper case}
Var
	Index : Word;   {Integer!}
	Found : Integer;
Begin
	Found := 0; {assume failure}

	For Index := 1 to NumConditions Do
	Begin
		If Conditions[Index] = Token Then
		Begin
			Found := Index;                  {found it}
			Break;   {Index := NumConditions;} {cancel the search}
		End;    {If}
	End; {For}

	IsCondition := Found;
End;    {IsCondition}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsAction(Token : String) : Integer;
{As 'IsCondition' above, this will take a token, and return a number greater
than 1 if it is a valid Action token, or 0 otherwise.

NOTE - again, the input string must be in upper case}
Var
	Index : Word;   {Integer!}
	Found : Integer;
Begin
	Found := 0; {assume failure}

	For Index := 1 to NumActions Do
	Begin
		If Actions[Index] = Token Then
		Begin
			Found := Index;                  {Found it}
			Break;   {Index := NumActions;}    {stop searching}
		End;    {If}
	End;    {For}

	IsAction := Found;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsCondAct(Token : String) : Integer;
{Function to check if token is valid - does it by calling IsCondition and
IsAction above}
Var
	Number : Integer;
Begin
	Number := IsAction(Token);	{check if it's an action}

	If Number > 0 Then 	{yes, it is}
	Begin
		IsCondAct := 100 + Number;
		Exit
	End Else Begin
		Number := IsCondition(Token);	{it isn't...check condition}

		If Number > 0 Then	{yes, condition}
		Begin
			IsCondAct := Number;
			Exit
		End;
	End;	{else}

	IsCondAct := 0;		{fall-through...return 'no'}
End; {IsCondAct}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function StringToList(InputString : PChar) : TList;
{Convert the input string, which should be a candidate table entry in the form:
	AT 1 GREATER 50 34 GOTO 4 MESSAGE 4 DONE
...into a list, with a string at each node, which will be either a token,
or a number....this makes validation easier}
Var
	CurrentWord    : String[11];
	Length, Index  : Word;  {Integer!}
	InputWordsList : TList;
	InputWord      : MGameInputWord;
Begin
	CurrentWord := '';
	Length := StrLen(InputString);

	InputWordsList := TList.Create; {Create the list}

	For Index := 0 to Length-1 Do
	Begin
		If InputString[Index] <> ' ' Then
			CurrentWord := CurrentWord + InputString[Index]
		Else Begin
			If CurrentWord <> '' Then       {This will strip whitespace}
			Begin
				InputWord := MGameInputWord.Create(CurrentWord);
					{create a valid object to put in the list}
				InputWordsList.Add(InputWord);  {add it to the list}
				CurrentWord := '';
			End;    {If currentword <> ''}
		End;    {Else}
	End;    {For}

	{At the end of the input string, we will have a word, so add it}

	If CurrentWord <> '' Then
	Begin
		InputWord := MGameInputWord.Create(CurrentWord);
		InputWordsList.Add(InputWord);
	End;    {If}

	StringToList := InputWordsList;
End; {StringToList}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidLocation(LocNumber : Word) : Boolean;
{Function to check if the number specified is a valid location - i.e. is there
a location which has that number?  Returns TRUE for YES, or FALSE otherwise}
Begin
	If ((FindLocation(LocNumber) <> 65535) Or
		(LocNumber = _CarriedLoc) Or
		(LocNumber = _WornLoc) Or
		(LocNumber = _NotCreatedLoc)) Then
		IsValidLocation := True
	Else
		IsValidLocation := False;
End;    {IsValidLocation}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidObject(ObjNumber : Word) : Boolean;
{Like IsValidLoction, this will check if the object number specified is valid.
Returns TRUE if an object with that number exists, or FALSE otherwise}
Begin
	IsValidObject := (Objects.FindObject(ObjNumber));
End;    {IsValidObject}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidMessage(MesNumber : Word) : Boolean;
{Will check if a message exists with the number specified.  Returns TRUE if so,
FALSE otherwise}
Begin
	If (FindMessage(MesNumber) = 65535) Then
		IsValidMessage := False
	Else
		IsValidMessage := True;
End;    {IsValidMessage}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidIntMess(IntMessNumber : Word) : Boolean;
{Will check if an Internal Message exists with the number specified.  Returns
TRUE if so, FALSE otherwise}
Begin
	If (IntMessages.FindMessage(IntMessNumber) = 65535) Then
		IsValidIntMess := False
	Else
		IsValidIntMess := True;
End;    {IsValidIntMess}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidTable(TableNumber : Word) : Boolean;
{Will check if the table number specified is valid}
Begin
	If (FindTable(TableNumber) = 65535) Then
		IsValidTable := False
	Else
		IsValidTable := True;
End;	{IsValidTable}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidNumber(Number : Word) : Boolean;
{Will check if the number specified is valid

Basically, if the number is less than 65536 (and ALL numbers which will fit
into a word are less than 65536!), then the number is valid.  So, just return
true.  Handy to have it check, though - makes expansion easier}
Begin
	IsValidNumber := True;
End;	{IsValidNumber}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidFlag(FlagNumber : Word) : Boolean;
{Will check if the flag number specified is valid}
Begin
	If FlagNumber <= NumFlags Then IsValidFlag := True
	Else IsValidFlag := False;
End;	{IsValidFlag}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidBulb(BulbNumber : Word) : Boolean;
{Checks if the bulb number specified is valid or not}
Begin
	If BulbNumber <= NumBulbs Then IsValidBulb := True
	Else IsValidBulb := False;
End;	{IsValidBulb}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidWord(WordValue : Word; TokenValue : Integer) : Boolean;
{Checks if the word specified is valid, i.e. are there any words of that type,
with that value, in the database}
Var
	Okay : Boolean;
Begin
	Okay := True;	{assume all will be okay}
	Case TokenValue Of
		16, 20 : Begin
			If (GetWordFromVocab(WordValue, Adjective) = '') Then
				Okay := False;
				End;
		17 : Begin
			If (GetWordFromVocab(WordValue, Adverb) = '') Then
				Okay := False;
			End;
		18 : Begin
			If (GetWordFromVocab(WordValue, Preposition) = '') Then
				Okay := False;
			End;
		19 : Begin
			If (GetWordFromVocab(WordValue, Noun) = '') Then
				Okay := False;
			End;
	End;

	IsValidWord := Okay;	{copy temp var to return var}
End;	{IsValidWord}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function ProcessInput(Input : TCaption; Var ErrorPosn : Integer;
						Verb, Noun : Word) : MGameTableEntry;
{The heart of the game designer.  It will take the input of the designer, and
parse it to check if the syntax is okay, i.e. the token 'at' should be followed
by 1 piece of numerical data, which should be a valid game location, etc.  Will
return the position of any error found, or 0 if the input was okay.  NOTE - this
does not check if the input will give the desired response, only that each
statement is valid.
Returns:	A TList, with the input given in tokenised form, i.e. all tokens
				are converted to integers, provided that the whole of the string
				is valid.  If there is one or more errors in the input, then
				the TList will be NIL - this should be checked by calling
				routine.
			An Integer, specified by the InOut variable ErrorPosn, which is an
				indication of where the error in the input string was discovered
				This is set to be just AFTER the erroneous token.  In this way
				a calling routine can put the cursor to where the error is, and
				inform the user.  If no errors found, returns 0}
Var
	TokenList : TList;      {A list to hold a split-down version of hte user's
						input - each node will hold a token - be that a word or
						a number - which can then be worked through to provide
						an integer-list}
	IntList : MGameTableEntry;
						{This is the integer list which will be returned to the
						calling routine.  It represents an integer
						representation of the textual input, with all tokens
						converted to their corresponding integer values, and
						data also converted from text to integers.}
	TempString : Array[0..255] Of Char;
	Index : Word;   {Integer!}
	ThisToken, TempToken : MGameInputWord;

	TokenPlace : Integer;	{the value of a particular token}
	NumData : Integer;	{The amount of data the token requires}
	IsAction : Boolean;	{Used to check if the condact is an action or not}
	Data1, Data2 : Integer;	{the next 2 tokens in the list}
	TempVal : Integer;
	Counter : Integer;

	ErrorPosition : Integer;	{The position in the input text where an error
								was found - that way, we can pass back where the
								error was, and then put the cursor just after
								the error, and beep or something, so the user
								knows he made a mistake}
Begin
	ErrorPosition := 0;
	TokenList := StringToList(StrPCopy(TempString,Input));

	NumData := 0;

	For Index := 0 To TokenList.Count -1 Do
	Begin
		If NumData <> 0 Then
		Begin
			Dec(NumData);	{this is here so that the loop will skip the data tokens}
		End Else Begin
			ThisToken:=TokenList.Items[Index];
			TokenPlace := IsCondAct(ThisToken.Token);
			If TokenPlace > 100 Then IsAction := True
				Else IsAction := False;
			If TokenPlace <> 0 Then
			Begin
				If IsAction = True Then	{get the amount of data it takes}
				Begin
					TokenPlace := TokenPlace - 100;
					NumData := StrToInt(ActDataVal[TokenPlace]);
					If NumData > 10 Then NumData := 2
					Else If NumData <> 0 Then NumData := 1
					Else NumData := 0;

					DebugWindow.AddText(ThisToken.Token + ' Action, '+
										IntToStr(NumData)+ ' Data Items')

				End Else Begin				{get the amount of data it takes}
					NumData := StrToInt(CondDataVal[TokenPlace]);
					If NumData > 10 Then NumData := 2
					Else If NumData <> 0 Then NumData := 1
					Else NumData := 0;

					DebugWindow.AddText(ThisToken.Token + ' Condition, '+
										IntToStr(NumData)+ ' Data Items');
				End;

				{now, we know whether it is a condition or an action, and how much
				data it takes - so go check the data}

				If NumData = 3 Then
				Begin
					DebugWindow.AddText('  ERROR - Invalid number of data items '+
										'reported');
					ErrorPosition := Index + 1 ;
				End;

				Data1 := 0;
				Data2 := 0;

				For Counter := 1 to NumData Do
				Begin
					If Index+Counter <= TokenList.Count-1 Then
					Begin
						TempToken := TokenList.Items[Index+Counter];
						TempVal := 0;
						Try
							TempVal := StrToInt(TempToken.Token);
						Except
							on Exception Do
							Begin
								DebugWindow.AddText(
									'    Error - wanted integer, got '+
									TempToken.Token);
								TempVal := 0;
								ErrorPosition := Index + Counter + 1;
							End;	{do}
						End;	{except}
						If Counter = 1 Then Data1 := TempVal
						Else Data2 := TempVal;
					End Else Begin
						DebugWindow.AddText('  ERROR - Not enough tokens for '+
											'required data');
						ErrorPosition := Index + 1;
					End;
				End;	{for counter}

				If (ValidateData(IsAction, TokenPlace, NumData, Data1, Data2))
						<> True Then ErrorPosition := Index + NumData + 1;

			End Else Begin
				DebugWindow.AddText(ThisToken.Token+' not matched');
				ErrorPosition := Index + 1
			End;
			If ErrorPosition <> 0 Then Break;	{Index := TokenList.Count -1;}
		End;
	End;    {For}

	DebugWindow.AddText('-----');

	If ErrorPosition = 0 Then	{No errors in input string}
	Begin
		IntList := TokensToIntegers(TokenList, Verb, Noun);

		ErrorPosn := 0;			{No errors}
		TokenList.Free;			{Release the memory}
		ProcessInput := IntList;	{Return the list}

	End Else Begin
		ErrorPosn := ErrorPosition;	{Return the position of the error}
		TokenList.Free;
		ProcessInput := Nil;			{And don't return a list}
	End;	{If}
End;    {ProcessInput}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function ValidateData(IsAction : Boolean;
						TokenValue : Integer;
						NumData, Data1, Data2 : Integer) : Boolean;
{function to check if the data following the token specified is valid}
Var
	DataSpec : String[2];
	ThisDigit : Integer;
	Index : Integer;
	Okay : Boolean;
	ThisDataS : String[11];
	ThisData : Integer;
Begin
	Okay := True;	{assume all is okay}
	If IsAction = True Then DataSpec := ActDataVal[TokenValue]
		Else DataSpec := CondDataVal[TokenValue];
	For Index := 1 to NumData Do
	Begin
		If NumData = 1 Then ThisDigit := Ord(DataSpec[2])-Ord('0')
		Else ThisDigit := Ord(DataSpec[Index])-Ord('0');
		If Index = 1 Then
		Begin
			ThisDataS := IntToStr(Data1);
			ThisData := Data1
		End Else Begin
			ThisDataS := IntToStr(Data2);
			ThisData := Data2;
		End;

		Case ThisDigit Of
			1 : If (IsValidLocation(ThisData) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad Location')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' Location OK');
			2 : If (IsValidMessage(ThisData) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad Message')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' Message OK');
			3 : If (IsValidIntMess(ThisData) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad IntMess')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' IntMess OK');
			4 : If (IsValidObject(ThisData) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad Object')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' Object OK');
			5 : If (IsValidTable(ThisData) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad Table')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' Table OK');
			6 : If (IsValidNumber(ThisData) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad Number')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' Number OK');
			7 : If (IsValidFlag(ThisData) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad Flag')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' Flag OK');
			8 : If (IsValidBulb(ThisData) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad Bulb')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' Bulb OK');
			9 : If (IsValidWord(ThisData, TokenValue) = False) Then
					Begin
						Okay := False;
						DebugWindow.AddText('  '+ThisDataS+' Bad Word')
					End Else
						DebugWindow.AddText('  '+ThisDataS+' Word OK');
		End;
	End;	{for}
	ValidateData := Okay;	{return status code}
End;	{ValidateData}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function TokensToIntegers(Input : TList; Verb, Noun : Word) : MGameTableEntry;
Var
	TempCondAct : MGameCondAct;
	TempEntry : MGameTableEntry;
	Index : Integer;
	ThisToken : MGameInputWord;
	TokenValue : Integer;
	NumData : Integer;
	Data1, Data2 : Word;
	Counter : Integer;
Begin
	TempEntry := MGameTableEntry.Create(Verb, Noun);

	DebugWindow.AddText(IntToStr(Input.Count)+' tokens');

	NumData := 0;

	For Index := 0 to Input.Count -1 Do
	Begin
		If NumData <> 0 Then
		Begin
			Dec(NumData);	{used to skip data tokens}
		End Else Begin
			DebugWindow.AddText('Index = '+IntToStr(Index+1));
			ThisToken := Input.Items[Index];
			TokenValue := IsCondAct(ThisToken.Token);

			If TokenValue > 100 Then NumData := StrToInt(ActDataVal[TokenValue-100])
			Else NumData := StrToInt(CondDataVal[TokenValue]);

			{Because of the way the ActDataVal and CondDataVal Arrays are
			constructed, the following statements are true for all tokens:

				1. If the value is 0, the token takes NO data.
				2. If the value is less than 10 (01-09), then the token takes one
					piece of data.
				3. Any other value means that the token takes two pieces of data.}

			If NumData > 10 Then NumData := 2
			Else If NumData <> 0 Then NumData := 1
			Else NumData := 0;

			Data1 := 0;
			Data2 := 0;

			DebugWindow.AddText('Token is '+ThisToken.Token+' with '+
									IntToStr(NumData)+' pieces of data');

			For Counter := 1 to NumData Do
			Begin
				ThisToken := Input.Items[Index+Counter];
				If Counter = 1 Then Data1 := StrToInt(ThisToken.Token)
				Else Data2 := StrToInt(ThisToken.Token);
			End;	{For Counter}

			TempCondAct := MGameCondAct.Create(TokenValue,Data1,Data2);

			AddCondActToEntry(TempEntry,TempCondAct);

			DebugWindow.AddText('Added');
		End;
	End;	{For}

	TokensToIntegers := TempEntry;
End;	{TokensToIntegers}




end.	{Text2Num}
