Unit VcbStuff;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
{* * *                                                                   * * * }
{ * * *  The purpose of this unit is to provide a series of routines      * * *}
{* * *    which deal with the vocabulary part of the database.  They     * * * }
{ * * *   provide functions which make manipulating the vocabulary        * * *}
{* * *    easier.                                                        * * * }
{ * * *                                                                   * * *}
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

{routines to make manipulating the vocabulary easier}

Interface

Uses
	Globals, Classes, SysUtils, Forms, WinTypes, ObjClash, ObjStuff, LocStuff;

Type
{valid word types recognised by the program}
	TypeOfWord=(NO_WORD, Verb, Noun, Adverb, Adjective, Preposition,
					Conjunction, Other);

{the structure used to hold our vocabulary}
	GameWord = Record
		Word : String[10];
		WordType : TypeOfWord;
		WordValue : Word;		{this is an integer type, with values from 0
									to 65535, we use 1 to 65500}
	End;

{re-define class TObject.  This has to be done, because the List class, TList,
holds objects of type TObject for a node.  So, we do this, and define that a
MGameVocab object is just a GameWord record}
	MGameVocab = Class(TObject)
		Vocabulary : GameWord;

		Constructor	Create(Word, WType, WValue : String);
		Function StringToWordType(WType : String) : TypeOfWord;
		Function WTypeToString(WType : TypeOfWord) : String;
	End;

	Function FindWordInVocab(ThisWord : String) : Boolean;
	Function GetPositionOfWord(ThisWord : String) : Word;	{returns an integer!}
	Function IsValidType(WordType : string) : Boolean;
	Function CheckObjects(ThisWord : String) : Boolean;
	Function CheckMovement(ThisWord : String) : Boolean;
	Function CheckActionTables(ThisWord : String) : Boolean;
	Function GetWordFromVocab(Value : Word; WType : TypeOfWord) : String;
	Function GetWordType(ThisWord : String) : TypeOfWord;
	Function GetWordValue(ThisWord : String) : Word;	{integer!}

	Function IsVerb(ThisWord : String) : Boolean;
	Function IsNoun(ThisWord : String) : Boolean;
	Function IsAdjective(ThisWord : String) : Boolean;
	Function IsAdverb(ThisWord : String) : Boolean;
	Function IsConjunction(ThisWord : String) : Boolean;
	Function IsPreposition(ThisWord : String) : Boolean;

Implementation

Uses
	VocabMnu, ObjMenu, LocMenu, ActStuff, Text2Num;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{* * *                                                                   * * * }
{* * *                   Functions start here!                           * * * }
{* * *                                                                   * * * }
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }




{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Constructor MGameVocab.Create(Word, WType, WValue : String);
{Note that no checks are made on the data here, and if they are needed, they
should be made elsewhere}
Begin
	Vocabulary.Word := Word;
	Vocabulary.WordType := StringToWordType(WType);
	Vocabulary.WordValue := StrToInt(WValue);
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameVocab.StringToWordType(WType : String) : TypeOfWord;
{convert the string entered into a valid WordType, as defined above}
Begin
	WType := LowerCase(WType);		{All word types are lowercase, with}
	Wtype[1] := UpCase(Wtype[1]);	{a capital first letter - then we know}
									{where we stand}
	If WType = 'Verb' Then StringToWordType := Verb
	Else If WType = 'Noun' Then StringToWordType := Noun
	Else If WType = 'Adverb' Then StringToWordType := Adverb
	Else If WType = 'Adjective' Then StringToWordType := Adjective
	Else If WType = 'Conjunction' Then StringToWordType := Conjunction
	Else If WType = 'Preposition' Then StringToWordType := Preposition
	Else If WType = 'Other' Then StringToWordType := Other
	Else StringToWordType := NO_WORD;	{to indicate an error occurred}
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function MGameVocab.WTypeToString(WType : TypeOfWord) : String;
{the opposite of the previous function.  Given a WordType, produce a string}
Begin
	If WType = Verb Then WtypeToString := 'Verb'
	Else If WType = Noun Then WtypeToString := 'Noun'
	Else If WType = Adverb Then WtypeToString := 'Adverb'
	Else If WType = Adjective Then WtypeToString := 'Adjective'
	Else If WType = Conjunction Then WtypeToString := 'Conjunction'
	Else If WType = Preposition Then WtypeToString := 'Preposition'
	Else If WType = Other Then WtypeToString := 'Other'
	Else WtypeToString := 'ERROR';	{to indicate an error occurred}
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function FindWordInVocab(ThisWord : String) : Boolean;
{Take a string, representing a word, and check if it is already in the vocab}
Var
	TempWord : MGameVocab;
	Index : Word;
	FoundWord : Boolean;
Begin
	FoundWord:=False;								{assume word not in vocab}
	If VocabList.Count <> 0 Then
		For Index:=0 to VocabList.Count-1 Do
		Begin
			TempWord:=VocabList.Items[Index];			{extract this object}
			If TempWord.Vocabulary.Word = ThisWord Then
				FoundWord:=True;						{word found}
		End;
	FindWordInVocab:=FoundWord;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetPositionOfWord(ThisWord : String) : Word;
{Take a string, and return the position of the word in the vocab.  Should use
	FindWordInVocab to ascertain whether it is actually there first!!

	If the word is not in the vocabulary, then the position will be returned as
		65535}
Var
	TempWord : MGameVocab;
	Index : Word;
	WordPosn : Word;   {Temp. var. to hold the position of the word}
Begin
	WordPosn := 65535;
	For Index:= 0 to VocabList.Count-1 Do
	Begin
		TempWord:=VocabList.Items[Index];			{extract this object}
		If TempWord.Vocabulary.Word = ThisWord Then
			WordPosn := Index;						{Update position variable}
	End;
	GetPositionOfWord := WordPosn;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsValidType(WordType : String) : Boolean;
{Take a string representing a type of word, e.g. 'Verb', and return TRUE if
	it IS a valid wordtype, or FALSE if not}
Const
	 WordTypes : Array[1..7] Of String = ('Verb', 'Noun', 'Adjective', 'Adverb',
			   'Conjunction', 'Preposition', 'Other');
Var
   Index : Word;
Begin
	 IsValidType:=False;      							{assume failure}
	 WordType:=LowerCase(WordType);						{string to lower case}
	 WordType[1]:=UpCase(WordType[1]);					{Capitalize first letter}
	 For Index:=1 To 7 Do
		 If WordType =WordTypes[Index] Then IsValidType:=True;	{okay...found}
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

function Synonyms(Vocab : GameWord) : Boolean;
{search through the vocabulary list to see if any word has the same value and
type as the word specified.  NOTE: have to check that any word found is NOT
the word specified!! :)}
Var
	Index : Word;	{integer!}
	ThisWord : MGameVocab;
	FoundOne : Boolean;		{to see if we have at least one synonym}
Begin
	FoundOne := False;
	For Index := 0 To VocabList.Count -1 Do
	Begin
		ThisWord := VocabList.Items[Index];
		If ((ThisWord.Vocabulary.WordType = Vocab.WordType) And
			(ThisWord.Vocabulary.WordValue = Vocab.WordValue) And
			(ThisWord.Vocabulary.Word <> Vocab.Word)) Then
				FoundOne := True;
	End;
	Synonyms := FoundOne;
End;

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function CheckObjects(ThisWord : String) : Boolean;
{will return : true : if the delete can go ahead
				false : to cancel the delete}
{Looks through the vocabulary list for the word specified, and notes its
number and type.  Then, checks through the list of objects (if it is a
noun or adjective), to see if it is used.  NOTE - only need to do this
if the word has no synonyms!}
Var
	ThisObject : GameObject;
	CurrentWord : MGameVocab;
	Index : Word;	{integer!}
	Clash : Boolean;	{used to check if the adj or noun is used}
	ObjShortName : String[50];
	ObjNum : Integer;
	CanContinue : Boolean;	{local copy of var showing permission to delete}
Begin
	{	1. is it a noun or adj?
		2. does it have any synonyms?
		3. do we have any objects?
		4. do we have a clash?
	}
	CanContinue := True;	{assume that the delete is okay}

	CurrentWord := VocabList.Items[GetPositionOfWord(ThisWord)];
	If ((CurrentWord.Vocabulary.WordType = Noun) Or
		(CurrentWord.Vocabulary.WordType = Adjective)) Then
	Begin
		If (Synonyms(CurrentWord.Vocabulary) = False) Then	{word has no synonyms}
		Begin
			If Objects.NumObjects <> 0 Then	{we have at least one object}
			Begin
				For Index := 0 to Objects.NumObjects -1 Do
				Begin
					ThisObject := Objects.getObjectAtIndex(Index);
					Clash := False;
					If (CurrentWord.Vocabulary.WordType = Noun) Then
					Begin
						If (CurrentWord.Vocabulary.WordValue = ThisObject.ObjNoun) Then
						Begin
							Clash := True;
							ObjShortName := ThisObject.ObjShortName;
							ObjNum := ThisObject.ObjNumber
						End
					End Else Begin	{adjective!}
						If (CurrentWord.Vocabulary.WordValue = ThisObject.ObjAdjective) Then
						Begin
							Clash := True;
							ObjShortName := ThisObject.ObjShortName;
							ObjNum := ThisObject.ObjNumber
						End
					End	{noun or adj?}
				End;	{for}
				If Clash = True Then
				Begin
					ClashDlg.ObjNum.Caption := IntToStr(ObjNum);
					ClashDlg.ObjName.Caption := ObjShortName;
					if ClashDlg.ShowModal = idNo Then
						CanContinue := False
					Else Begin	{can delete the linkes between object and word}
						For Index := 0 to Objects.NumObjects -1 Do
						Begin
							ThisObject := Objects.GetObjectAtIndex(Index);
							If ThisObject.ObjNoun = CurrentWord.Vocabulary.WordValue Then
								Objects.ResetNounAtIndex(Index);
							If ThisObject.ObjAdjective = CurrentWord.Vocabulary.WordValue Then
								Objects.ResetAdjectiveAtIndex(Index);
						End;	{for index}
						Application.MessageBox('If the noun or adjective field of an object referenced this word,'+#10+#13+
												'it has been reset.',
												'Information',
												mb_OK + mb_ICONINFORMATION);
					End;	{showmodal = idYES}

				End;	{if clash = true}
			End	{object list.count <> 0}
		End	{synonyms}
	End;	{noun or adj}
	CheckObjects := CanContinue;
End;	{checkobjects}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function CheckMovement (ThisWord : String) : Boolean;
{check the connections table for each location and see if the word we are about
to delete is a connection.  If so, prompt to destroy the link.  If user presses
'yes', then destroy the link, if not, return a code meanung deletion of this word
should not happen}
{	1. Is word a noun?
	2. Is it a movement noun? 	(i.e does it have value < 14?)
	3. Does it have any synonyms?
	4. Is it used?
	5. Can it be deleted	( <- ask user)
}
Var
	Clash, ContinueOK : Boolean;
	CurrentWord : MGameVocab;
	ThisLocation : MGameLocation;
	SpareList : TList;
	Index : Word;
Begin
	Clash := False;
	ContinueOK := True;	{assume the delete can go ahead}

	CurrentWord := VocabList.Items[GetPositionOfWord(ThisWord)];

	If ((CurrentWord.Vocabulary.WordType = Noun) And
		(CurrentWord.Vocabulary.WordValue < 14)) Then
	Begin
		If (Synonyms(CurrentWord.Vocabulary) = False) Then
		Begin
			If LocationList.Count <> 0 Then
			Begin
				For Index := 0 to LocationList.Count -1 Do
				Begin
					ThisLocation := LocationList.Items[Index];

					If ThisLocation.Location.Exits[CurrentWord.Vocabulary.WordValue] <> 0 Then
						Clash := True;
				End;	{for index}

				If Clash = True Then
				Begin
					If (Application.MessageBox('The word you are about to delete is a movement noun,'+#10+#13+
												'which has no synonyms, and is used to indicate an exit'+#10+#13+
												'from one or more locations, in the connections table.'+#10+#10+#13+
												'Deleting this word would reset the connections.'+#10+#10+#13+
												'Do you want to delete this word?',
												'Question',
												MB_YESNO + MB_ICONQUESTION) = id_No) Then
						ContinueOK := False
					Else Begin
						SpareList := TList.Create;

						For Index := 0 to LocationList.Count -1 Do
						Begin
							ThisLocation := LocationList.Items[Index];

							ThisLocation.Location.Exits[CurrentWord.Vocabulary.WordValue] := 0;

							SpareList.Add(ThisLocation);
						End;	{for index}

						LocationList.Free;

						LocationList := TList.Create;

						For Index := 0 to SpareList.Count -1 Do
						Begin
							ThisLocation := SpareList.Items[index];

							LocationList.Add(ThisLocation);
						End;	{for index}

						SpareList.Free;

						SortLocationList;

						Application.MessageBox('Any connections which used this word have been reset',
												'Information',
												mb_OK + mb_ICONINFORMATION);
					End;	{user pressed 'yes' to delete the word....so delete the connections}
				End;	{clash = true}
			End;	{we have at least one location}
		End;	{word has synonyms}
	End;	{word is noun and val < 14}
	CheckMovement := ContinueOK;
End;	{CheckMovement}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function CheckActionTables(ThisWord : String) : Boolean;
{will search the action tables to see if the word specified is used
Returns:	True if the word is found at least once in the action tables
			False if the word is not found}
Var
	Found : Boolean;
	WordVal : Word;	{integer}
	TableCounter, EntryCounter, CondActCounter : Word;
	ThisTable : MGameTable;
	ThisEntry : MGameTableEntry;
	ThisCondAct : MGameCondAct;
	NumEntries : Word;
	NumCondActs : Word;
Begin
	Found := False;		{assume word not in tables}

	WordVal := GetWordValue(ThisWord);

	If WordVal <> 65535 Then
	Begin
		If (HowManyTables <> 0) Then
		Begin
			For TableCounter := 0 to HowManyTables -1 Do
			Begin
				ThisTable := GetTableFromList(TableCounter);

				NumEntries := ThisTable.NumEntries;

				If (NumEntries <> 0) Then
				Begin
					For EntryCounter := 0 To NumEntries -1 Do
					Begin
						ThisEntry := ThisTable.Table.Items[EntryCounter];

						If ((ThisEntry.Verb = WordVal) Or
							(ThisEntry.Noun = WordVal)) Then
						Begin
							Found := True;		{found it}
{							EntryCounter := NumEntries -1;}	{cancel the search}
{							TableCounter := HowManyTables -1;}
						End;	{word found}

						NumCondActs := ThisEntry.NumCondActs;

						If NumCondActs <> 0 Then
						Begin
							For CondActCounter := 0 to NumCondActs -1 Do
							Begin
								ThisCondAct := ThisEntry.TableEntry.Items[CondActCounter];
								{the only tokens which are followed by word values are these:
									ADJECT1, ADVERB, PREP, NOUN2, ADJECT2
									  16		17	  18	19		20}
								If ((ThisCondAct.CondAct.Token > 15) And
									(ThisCondAct.CondAct.Token < 21) And
									(ThisCondAct.CondAct.Data1 = WordVal)) Then	{matched}
										Found := True;
							End;	{for numcondacts}
						End;	{if numcondacts}
					End;	{For entryCounter}
				End;	{if numentries}
			End;	{For}
		End;	{if no tables}
	End;	{if wordval valid}

	{when we get to here, if found = false then the word to be deleted is not in
	the action tables.  If, however, found = true, then it is in the tables, so
	prompt the user if they really want to delete the word, and delete any
	offending entries, if so.  If not, exit gracefully.}

	If (Found = True) Then
	Begin
		If (Application.MessageBox('The word you are about to delete is used in one or'+#10+#13+
									'more of the Action Tables in this game, either as'+#10+#13+
									'the verb or noun of an entry, or is referenced by'+#10+#13+
									'one of the Vocabulary conditions, such as ''ADJECT1'''+#10+#13+
									'Deleting this word would delete the entries concerned.'+#10+#10+#13+
									'Do you want to delete this word?',
									'Question',
									MB_YESNO + MB_ICONQUESTION) = id_No) Then
		Else Begin
			For TableCounter := 0 to HowManyTables -1 Do
			Begin
				ThisTable := GetTableFromList(TableCounter); {TableList.Items[TableCounter];}

				NumEntries := ThisTable.NumEntries;

				If (NumEntries <> 0) Then
				Begin
					For EntryCounter := 0 To NumEntries -1 Do
					Begin
						ThisEntry := ThisTable.Table.Items[EntryCounter];

						If ((ThisEntry.Verb = WordVal) Or
							(ThisEntry.Noun = WordVal)) Then
						Begin
							Found := True;		{found it}
{							EntryCounter := NumEntries -1;}	{cancel the search}
{							TableCounter := HowManyTables -1;}
						End;	{word found}

						NumCondActs := ThisEntry.NumCondActs;

						If NumCondActs <> 0 Then
						Begin
							For CondActCounter := 0 to NumCondActs -1 Do
							Begin
								ThisCondAct := ThisEntry.TableEntry.Items[CondActCounter];
								{the only tokens which are followed by word values are these:
									ADJECT1, ADVERB, PREP, NOUN2, ADJECT2
									  16		17	  18	19		20}
								If ((ThisCondAct.CondAct.Token > 15) And
									(ThisCondAct.CondAct.Token < 21) And
									(ThisCondAct.CondAct.Data1 = WordVal)) Then	{matched}
										Found := True;
							End;	{for numcondacts}
						End;	{if numcondacts}

						If Found = True Then
						Begin
							DeleteEntryFromList(TableCounter, EntryCounter);
{							EntryCounter := 0;}
						End;	{If}
					End;	{For entryCounter}
				End;	{if numentries}
			End;	{For}

		End;	{user pressed 'yes' to delete the word....so delete the affected
					table entries}
		CheckActionTables := True;
	End Else Begin
		CheckActionTables := True;	{exit}
	End;	{if found = true}
End;	{CheckActionTables}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetWordFromVocab(Value : Word; WType : TypeOfWord) : String;
{returns the first word in the vocab which has the value and word type
	specified}
Var
	Index : Word;	{integer!}
	ThisWord : MGameVocab;
Begin
	If VocabList.Count = 0 Then
		GetWordFromVocab := '';

	For Index := 0 to VocabList.Count -1 Do
	Begin
		ThisWord := VocabList.Items[Index];

		If (ThisWord.Vocabulary.WordType = WType) And
			(ThisWord.Vocabulary.WordValue = Value) Then
			GetWordFromVocab := ThisWord.Vocabulary.Word
	End;	{for}
End;	{get word from vocab}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetWordType(ThisWord : String) : TypeOfWord;
{return the type of word that is associated with the word ThisWord.
Does not check whether the word is actually in the vocab!
Returns : The type of word.  If the word is not in the vocab, the word type
			will be set to NO_WORD}
Var
	WordPosn : Word;	{Integer!}
	ThisVocabWord : MGameVocab;
Begin
	WordPosn := GetPositionOfWord(ThisWord);

	If WordPosn <> 65535 Then
	Begin
		ThisVocabWord := VocabList.Items[WordPosn];

		GetWordType := ThisVocabWord.Vocabulary.WordType
	End Else Begin
		GetWordType := NO_WORD;
	End;	{If}
End;	{GetWordType}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function GetWordValue(ThisWord : String) : Word;	{Integer!}
{returns the value of the word specified.  Does not check if the word is
in the vocabulary - this should be done also.

Returns:	The value of the word specified.  If the word is not in the
			vocabulary, the value will be returned as 65535}
Var
	WordPosn : Word;	{Integer!}
	ThisVocabWord : MGameVocab;
Begin
	WordPosn := GetPositionOfWord(ThisWord);

	If WordPosn <> 65535 Then
	Begin
		ThisVocabWord := VocabList.Items[WordPosn];

		GetWordValue := ThisVocabWord.Vocabulary.WordValue
	End Else Begin
		GetWordValue := 65535;
	End;	{If}
End;	{GetWordValue}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsVerb(ThisWord : String) : Boolean;
Var
	WordPosn : Word;	{Integer!}
	ThisVcb : MGameVocab;
Begin
	If (FindWordInVocab(ThisWord) = True) Then
	Begin
		WordPosn := GetPositionOfWord(ThisWord);
		ThisVcb := VocabList.Items[WordPosn];
		If (ThisVcb.Vocabulary.WordType = Verb) Then
			IsVerb := True
		Else
			IsVerb := False;
	End Else Begin
		IsVerb := False;
	End;	{if valid word}
End;	{IsVerb}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsNoun(ThisWord : String) : Boolean;
Var
	WordPosn : Word;	{Integer!}
	ThisVcb : MGameVocab;
Begin
	If (FindWordInVocab(ThisWord) = True) Then
	Begin
		WordPosn := GetPositionOfWord(ThisWord);
		ThisVcb := VocabList.Items[WordPosn];
		If (ThisVcb.Vocabulary.WordType = Noun) Then
			IsNoun := True
		Else
			IsNoun := False;
	End Else Begin
		IsNoun := False;
	End;	{if valid word}
End;	{IsNoun}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsAdjective(ThisWord : String) : Boolean;
Var
	WordPosn : Word;	{Integer!}
	ThisVcb : MGameVocab;
Begin
	If (FindWordInVocab(ThisWord) = True) Then
	Begin
		WordPosn := GetPositionOfWord(ThisWord);
		ThisVcb := VocabList.Items[WordPosn];
		If (ThisVcb.Vocabulary.WordType = Adjective) Then
			IsAdjective := True
		Else
			IsAdjective := False;
	End Else Begin
		IsAdjective := False;
	End;	{if valid word}
End;	{IsAdjective}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsAdverb(ThisWord : String) : Boolean;
Var
	WordPosn : Word;	{Integer!}
	ThisVcb : MGameVocab;
Begin
	If (FindWordInVocab(ThisWord) = True) Then
	Begin
		WordPosn := GetPositionOfWord(ThisWord);
		ThisVcb := VocabList.Items[WordPosn];
		If (ThisVcb.Vocabulary.WordType = Adverb) Then
			IsAdverb := True
		Else
			IsAdverb := False;
	End Else Begin
		IsAdverb := False;
	End;	{if valid word}
End;	{IsAdverb}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsConjunction(ThisWord : String) : Boolean;
Var
	WordPosn : Word;	{Integer!}
	ThisVcb : MGameVocab;
Begin
	If (FindWordInVocab(ThisWord) = True) Then
	Begin
		WordPosn := GetPositionOfWord(ThisWord);
		ThisVcb := VocabList.Items[WordPosn];
		If (ThisVcb.Vocabulary.WordType = Conjunction) Then
			IsConjunction := True
		Else
			IsConjunction := False;
	End Else Begin
		IsConjunction := False;
	End;	{if valid word}
End;	{IsConjunction}

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * }
{ * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

Function IsPreposition(ThisWord : String) : Boolean;
Var
	WordPosn : Word;	{Integer!}
	ThisVcb : MGameVocab;
Begin
	If (FindWordInVocab(ThisWord) = True) Then
	Begin
		WordPosn := GetPositionOfWord(ThisWord);
		ThisVcb := VocabList.Items[WordPosn];
		If (ThisVcb.Vocabulary.WordType = Preposition) Then
			IsPreposition := True
		Else
			IsPreposition := False;
	End Else Begin
		IsPreposition := False;
	End;	{if valid word}
End;	{IsPreposition}

End.	{VcbStuff}
